// SPDX-FileCopyrightText: 2023 g10 code GmbH
// SPDX-Contributor: Carl Schwan <carl.schwan@gnupg.com>
// SPDX-License-Identifier: GPL-2.0-or-later

#include <QCommandLineParser>
#include <QCoreApplication>
#include <QTemporaryDir>

#include <KLocalizedString>

#include "controller.h"
#include <iostream>

#ifdef Q_OS_WINDOWS
#include <windows.h>
#endif

using namespace Qt::StringLiterals;

int main(int argc, char *argv[])
{
    QCoreApplication app(argc, argv);
    app.setApplicationName(QStringLiteral("GpgOL/Web cert generator"));
    app.setApplicationVersion(QStringLiteral("0.0.1"));
    app.setOrganizationName(QStringLiteral("g10code"));
    app.setOrganizationDomain(QStringLiteral("gnupg.com"));
    KLocalizedString::setApplicationDomain(QByteArrayLiteral("gpgol"));
    QCommandLineParser parser;

    auto helpOpt = parser.addHelpOption();
    parser.setApplicationDescription(i18n("Helper application to generate certificates for GgpOL/Web\nWithout any arguments, a certificate set is generated"));

    parser.process(app);
    if (parser.isSet(helpOpt)) {
        std::cout << qPrintable(parser.helpText());
        return 0;
    }
    QTemporaryDir tempDir;
    tempDir.setAutoRemove(false);

#ifdef Q_OS_WINDOWS
    if (AttachConsole(ATTACH_PARENT_PROCESS)) {
        freopen("CONOUT$", "w", stdout);
        freopen("CONOUT$", "w", stderr);
    }
#endif

    qWarning() << tempDir.path();

    qputenv("GNUPGHOME", tempDir.path().toUtf8());

    auto controller = new Controller;
    QObject::connect(controller, &KJob::finished, &app, [&app](KJob *job) {
        if (job->error()) {
            app.exit(1);
        } else {
            app.exit(0);
        }
    });
    controller->start();

    return app.exec();
}