// SPDX-FileCopyrightText: 2023 g10 code Gmbh
// SPDX-Contributor: Carl Schwan <carl.schwan@gnupg.com>
// SPDX-License-Identifier: GPL-2.0-or-later

#include <QApplication>
#include <QCommandLineParser>
#include <QElapsedTimer>
#include <QFile>
#include <QHttpServer>
#include <QHttpServerResponse>
#include <QJsonArray>
#include <QJsonDocument>
#include <QJsonObject>
#include <QNetworkReply>
#include <QPointer>
#include <QTimer>
#include <QUuid>

#include <Libkleo/FileSystemWatcher>
#include <Libkleo/GnuPG>
#include <Libkleo/KeyCache>

#include <KAboutData>
#include <KJob>
#include <KLocalizedString>

#include "../common/log.h"
#include "config.h"
#include "firsttimedialog.h"
#include "gpgol_client_debug.h"
#include "gpgolweb_version.h"
#include "utils/kuniqueservice.h"

using namespace Qt::Literals::StringLiterals;
using namespace std::chrono;

#ifdef Q_OS_WINDOWS
#include <windows.h>
#endif

#define STARTUP_TIMING qCDebug(GPGOL_CLIENT_LOG) << "Startup timing:" << startupTimer.elapsed() << "ms:"
#define STARTUP_TRACE qCDebug(GPGOL_CLIENT_LOG) << "Startup timing:" << startupTimer.elapsed() << "ms:" << SRCNAME << __func__ << __LINE__;

static void setupLogging(std::shared_ptr<Kleo::Log> log)
{
    const QByteArray dirNative = qgetenv("GPGOL_CLIENT_LOGDIR");
    if (dirNative.isEmpty()) {
        return;
    }
    log->setOutputDirectory(QFile::decodeName(dirNative));
    qInstallMessageHandler(Kleo::Log::messageHandler);
}

int main(int argc, char *argv[])
{
#ifdef Q_OS_WINDOWS
    if (AttachConsole(ATTACH_PARENT_PROCESS)) {
        freopen("CONOUT$", "w", stdout);
        freopen("CONOUT$", "w", stderr);
    }
#endif

    QApplication app(argc, argv);
    app.setQuitOnLastWindowClosed(false);
    // On Windows, broken heuristics appear to be in place WRT whether our app should exit
    // when the last QEventLocker goes out of scope, resulting in sudden quits, after handling a
    // websocket request while minimized to systray (the specific conditions appear to include that no
    // other window besides the FirstTimeDialog had been visible before minimzing).
    // We prefer to quit on our own terms, thank you!
    app.setQuitLockEnabled(false);

    KLocalizedString::setApplicationDomain(QByteArrayLiteral("gpgol-js-native"));

    KAboutData about(QStringLiteral("gpgol-client"),
                     i18nc("@title:window", "GnuPG Outlook Add-in"),
                     QStringLiteral(GPGOLWEB_VERSION_STRING),
                     i18nc("@info", "GPG Outlook add-in"),
                     KAboutLicense::GPL,
                     i18nc("@info:credit", "© 2023-2025 g10 Code GmbH"));

    about.setDesktopFileName(u"com.gnupg.gpgolweb"_s);
    about.setProgramLogo(QIcon::fromTheme(u"com.gnupg.gpgolweb"_s));

    about.addAuthor(i18nc("@info:credit", "Carl Schwan"), i18nc("@info:credit", "Maintainer"), u"carl.schwan@gnupg.com"_s, u"https://carlschwan.eu"_s);

    about.setTranslator(i18nc("NAME OF TRANSLATORS", "Your names"), i18nc("EMAIL OF TRANSLATORS", "Your emails"));

    about.setOrganizationDomain("gnupg.com");
    about.setBugAddress("https://dev.gnupg.org/maniphest/task/edit/form/1/");

    QCommandLineParser parser;

    KAboutData::setApplicationData(about);

    about.setupCommandLine(&parser);
    parser.process(app);
    about.processCommandLine(&parser);

    QElapsedTimer startupTimer;
    startupTimer.start();

    STARTUP_TIMING << "Application created";
    /* Create the unique service ASAP to prevent double starts if
     * the application is started twice very quickly. */
    KUniqueService service;

    QObject::connect(&service, &KUniqueService::activateRequested, &service, [&service](const QStringList &arguments, const QString &workingDirectory) {
        Q_UNUSED(arguments);
        Q_UNUSED(workingDirectory);
        service.setExitValue(0);
    });

    auto log = Kleo::Log::mutableInstance();
    setupLogging(log);

    STARTUP_TIMING << "Service created";

    QPointer<FirstTimeDialog> launcher = new FirstTimeDialog;

    if (Config::self()->showLauncher()) {
        launcher->show();
    }

    STARTUP_TIMING << "KeyCache creation";
    auto keyCache = Kleo::KeyCache::mutableInstance();
    auto fsWatcher = std::make_shared<Kleo::FileSystemWatcher>();

    fsWatcher->whitelistFiles(Kleo::gnupgFileWhitelist());
    fsWatcher->addPaths(Kleo::gnupgFolderWhitelist());
    fsWatcher->setDelay(1000);
    keyCache->addFileSystemWatcher(fsWatcher);

    keyCache->startKeyListing();

    return app.exec();
}
