// SPDX-FileCopyrightText: 2023 g10 code Gmbh
// SPDX-Contributor: Carl Schwan <carl.schwan@gnupg.com>
// SPDX-License-Identifier: GPL-2.0-or-later

#include "websocketclient.h"

// Qt headers
#include <QFile>
#include <QHostInfo>
#include <QJsonArray>
#include <QJsonDocument>
#include <QJsonObject>
#include <QStandardPaths>
#include <QTimer>
#include <QUuid>

// KDE headers
#include <KConfigGroup>
#include <KLocalizedString>
#include <KMime/Message>
#include <KSharedConfig>
#include <Libkleo/Formatting>
#include <Libkleo/KeyCache>
#include <MimeTreeParserCore/ObjectTreeParser>

// gpgme headers
#include <QGpgME/KeyListJob>
#include <QGpgME/Protocol>
#include <gpgme++/global.h>
#include <gpgme++/key.h>
#include <gpgme++/keylistresult.h>

#include "config.h"
#include "draft/draftmanager.h"
#include "editor/composerwindow.h"
#include "editor/composerwindowfactory.h"
#include "emailviewer.h"
#include "ews/ewsclient.h"
#include "ews/ewsgetitemrequest.h"
#include "ews/ewsitemshape.h"
#include "gpgol_client_debug.h"
#include "gpgolweb_version.h"
#include "protocol.h"
#include "reencrypt/reencryptjob.h"
#include "websocket_debug.h"

using namespace Qt::Literals::StringLiterals;

WebsocketClient &WebsocketClient::self(const QUrl &url, const QString &clientId)
{
    static WebsocketClient *client = nullptr;
    if (!client && url.isEmpty()) {
        qFatal() << "Unable to create a client without an url";
    } else if (!client) {
        client = new WebsocketClient(url, clientId);
    }
    return *client;
};

WebsocketClient::WebsocketClient(const QUrl &url, const QString &clientId)
    : QObject(nullptr)
    , m_webSocket(QWebSocket(QStringLiteral("Client")))
    , m_url(url)
    , m_clientId(clientId)
    , m_state(NotConnected)
    , m_stateDisplay(i18nc("@info", "Loading..."))
{
    auto job = QGpgME::openpgp()->keyListJob();
    connect(job, &QGpgME::KeyListJob::result, this, &WebsocketClient::slotKeyListingDone);
    job->start({}, true);

    qCWarning(GPGOL_CLIENT_LOG) << "Found the following trusted emails" << m_emails;

    connect(&m_webSocket, &QWebSocket::connected, this, &WebsocketClient::slotConnected);
    connect(&m_webSocket, &QWebSocket::disconnected, this, [this] {
        m_state = NotConnected;
        m_stateDisplay = i18nc("@info", "Connection to outlook lost due to a disconnection with the broker.");
        Q_EMIT stateChanged(m_stateDisplay);
    });
    connect(&m_webSocket, &QWebSocket::errorOccurred, this, &WebsocketClient::slotErrorOccurred);
    connect(&m_webSocket, &QWebSocket::textMessageReceived, this, &WebsocketClient::slotTextMessageReceived);
    connect(&m_webSocket, QOverload<const QList<QSslError> &>::of(&QWebSocket::sslErrors), this, [this](const QList<QSslError> &errors) {
        // TODO remove
        m_webSocket.ignoreSslErrors(errors);
    });

    QSslConfiguration sslConfiguration;
    auto certPath = QStandardPaths::locate(QStandardPaths::AppLocalDataLocation, QStringLiteral("certificate.pem"));
    Q_ASSERT(!certPath.isEmpty());

    QFile certFile(certPath);
    if (!certFile.open(QIODevice::ReadOnly)) {
        qFatal() << "Couldn't read certificate" << certPath;
    }
    QSslCertificate certificate(&certFile, QSsl::Pem);
    certFile.close();
    sslConfiguration.addCaCertificate(certificate);
    m_webSocket.setSslConfiguration(sslConfiguration);

    m_webSocket.open(url);
}

void WebsocketClient::slotKeyListingDone(const GpgME::KeyListResult &result, const std::vector<GpgME::Key> &keys, const QString &, const GpgME::Error &error)
{
    Q_UNUSED(result);
    Q_UNUSED(error);

    if (error) {
        m_stateDisplay = i18nc("@info", "Connection to the Outlook extension lost. Make sure the extension pane is open.");
        Q_EMIT stateChanged(m_stateDisplay);
        return;
    }

    QStringList oldEmails = m_emails;

    for (const auto &key : keys) {
        for (const auto &userId : key.userIDs()) {
            const auto email = QString::fromLatin1(userId.email()).toLower();
            if (!m_emails.contains(email)) {
                m_emails << email;
            }
        }
    }
    if (m_emails == oldEmails) {
        return;
    }

    qCWarning(GPGOL_CLIENT_LOG) << "Found the following trusted emails" << m_emails;

    if (m_webSocket.state() == QAbstractSocket::ConnectedState) {
        slotConnected();
    }
}

void WebsocketClient::slotConnected()
{
    qCInfo(WEBSOCKET_LOG) << "websocket connected";
    // clang-format off
    QJsonDocument doc(QJsonObject{
        {"command"_L1, Protocol::commandToString(Protocol::Register)},
        {"arguments"_L1, QJsonObject{
            {"emails"_L1, QJsonArray::fromStringList(m_emails)},
            {"type"_L1, "native"_L1},
            {"id"_L1, getId() },
            {"name"_L1, QString(QHostInfo::localHostName() + u" - GpgOL/Web ("_s + QStringLiteral(GPGOLWEB_VERSION_STRING) + u')') },
        }},
    });
    // clang-format on

    m_webSocket.sendTextMessage(QString::fromUtf8(doc.toJson()));

    m_state = NotConnected; /// We still need to connect to the web client
    m_stateDisplay = i18nc("@info", "Waiting for web client.");
    Q_EMIT stateChanged(m_stateDisplay);
}

void WebsocketClient::slotErrorOccurred(QAbstractSocket::SocketError error)
{
    qCWarning(WEBSOCKET_LOG) << error << m_webSocket.errorString();
    m_state = NotConnected;
    m_stateDisplay = i18nc("@info", "Could not reach the Outlook extension.");
    Q_EMIT stateChanged(m_stateDisplay);
    reconnect();
}

bool WebsocketClient::sendEWSRequest(const QString &fromEmail, const QString &requestId, const QString &requestBody)
{
    KMime::Types::Mailbox mailbox;
    mailbox.fromUnicodeString(fromEmail);

    const QJsonObject json{
        {"command"_L1, Protocol::commandToString(Protocol::Ews)},
        {"arguments"_L1,
         QJsonObject{
             {"body"_L1, requestBody},
             {"email"_L1, QString::fromUtf8(mailbox.address())},
             {"requestId"_L1, requestId},
         }},
    };

    m_webSocket.sendTextMessage(QString::fromUtf8(QJsonDocument(json).toJson()));
    return true;
}

void WebsocketClient::slotTextMessageReceived(QString message)
{
    const auto doc = QJsonDocument::fromJson(message.toUtf8());
    if (!doc.isObject()) {
        qCWarning(WEBSOCKET_LOG) << "invalid text message received" << message;
        return;
    }

    const auto object = doc.object();
    const auto command = Protocol::commandFromString(object["command"_L1].toString());
    const auto args = object["arguments"_L1].toObject();

    switch (command) {
    case Protocol::Disconnection:
        // disconnection of the web client
        m_state = NotConnected;
        m_stateDisplay = i18nc("@info", "Connection to the Outlook extension lost. Make sure the extension pane is open.");
        Q_EMIT stateChanged(m_stateDisplay);
        return;
    case Protocol::Connection:
        // reconnection of the web client
        m_state = Connected;
        m_stateDisplay = i18nc("@info", "Connected.");
        Q_EMIT stateChanged(m_stateDisplay);
        return;
    case Protocol::View: {
        const auto email = args["email"_L1].toString();
        const auto displayName = args["displayName"_L1].toString();
        const EwsId id(args["itemId"_L1].toString());
        const auto content = m_cachedMime[id];
        if (content.isEmpty()) {
            return;
        }
        const auto ewsAccessToken = args["ewsAccessToken"_L1].toString().toUtf8();

        if (!m_emailViewer) {
            m_emailViewer = new EmailViewer(content, email, displayName, ewsAccessToken);
            m_emailViewer->setAttribute(Qt::WA_DeleteOnClose);
        } else {
            m_emailViewer->view(content, email, displayName, ewsAccessToken);
        }

        m_emailViewer->show();
        m_emailViewer->activateWindow();
        m_emailViewer->raise();
        return;
    }
    case Protocol::RestoreAutosave: {
        const auto email = args["email"_L1].toString();
        const auto displayName = args["displayName"_L1].toString();
        const auto ewsAccessToken = args["ewsAccessToken"_L1].toString().toUtf8();

        ComposerWindowFactory::self().restoreAutosave(email, displayName, ewsAccessToken);
        return;
    }
    case Protocol::EwsResponse: {
        // confirmation that the email was sent
        const auto args = object["arguments"_L1].toObject();
        Q_EMIT ewsResponseReceived(args["requestId"_L1].toString(), args["body"_L1].toString());
        return;
    }
    case Protocol::Composer:
    case Protocol::Reply:
    case Protocol::Forward:
    case Protocol::OpenDraft: {
        const auto email = args["email"_L1].toString();
        const auto displayName = args["displayName"_L1].toString();
        const auto ewsAccessToken = args["ewsAccessToken"_L1].toString().toUtf8();

        auto dialog = ComposerWindowFactory::self().create(email, displayName, ewsAccessToken);

        if (command == Protocol::Reply || command == Protocol::Forward) {
            const EwsId id(args["itemId"_L1].toString());
            const auto content = m_cachedMime[id];
            if (content.isEmpty()) {
                return;
            }

            KMime::Message::Ptr message(new KMime::Message());
            message->setContent(KMime::CRLFtoLF(content.toUtf8()));
            message->parse();
            if (command == Protocol::Reply) {
                dialog->reply(message);
            } else {
                dialog->forward(message);
            }
        } else if (command == Protocol::OpenDraft) {
            const auto draftId = args["id"_L1].toString();
            if (draftId.isEmpty()) {
                return;
            }
            const auto draft = DraftManager::self().draftById(draftId.toUtf8());
            dialog->setMessage(draft.mime());
        }
        dialog->show();
        dialog->activateWindow();
        dialog->raise();
        return;
    }
    case Protocol::DeleteDraft: {
        const auto draftId = args["id"_L1].toString();
        if (draftId.isEmpty()) {
            qWarning() << "Draft not valid";
            return;
        }
        const auto draft = DraftManager::self().draftById(draftId.toUtf8());

        if (!draft.isValid()) {
            qWarning() << "Draft not valid";
            return;
        }

        if (!DraftManager::self().remove(draft)) {
            qCWarning(GPGOL_CLIENT_LOG) << "Could not delete draft";
            return;
        }
        sendStatusUpdate(args["email"_L1].toString());
        return;
    }
    case Protocol::Reencrypt: {
        reencrypt(args);
        return;
    }
    case Protocol::Info: {
        info(args);
        return;
    }
    default:
        qCWarning(WEBSOCKET_LOG) << "Unhandled command" << command;
    }
}

void WebsocketClient::reencrypt(const QJsonObject &args)
{
    const EwsId folderId(args["folderId"_L1].toString());

    EwsClient client(args["email"_L1].toString(), args["ewsAccessToken"_L1].toString().toUtf8());

    if (m_reencryptJob) {
        if (m_reencryptJob->hasStarted()) {
            m_reencryptJob->tryRaiseDialog();
            return;
        }
        m_reencryptJob->deleteLater();
    }

    m_reencryptJob = new ReencryptJob(this, folderId, client);
    m_reencryptJob->start();
}

void WebsocketClient::reconnect()
{
    QTimer::singleShot(1000ms, this, [this]() {
        m_webSocket.open(m_url);
    });
}

WebsocketClient::State WebsocketClient::state() const
{
    return m_state;
}

QString WebsocketClient::stateDisplay() const
{
    return m_stateDisplay;
}

void WebsocketClient::sendStatusUpdate(const QString &email, bool viewerJustClosed)
{
    QJsonArray features;
    if (Config::self()->reencrypt()) {
        features << u"reencrypt"_s;
    }
    const QJsonObject json{{"command"_L1, Protocol::commandToString(Protocol::StatusUpdate)},
                           {"arguments"_L1, QJsonObject{
                                {"email"_L1, email},
                                {"drafts"_L1, DraftManager::self().toJson()},
                                {"viewerOpen"_L1, !viewerJustClosed && !m_emailViewer.isNull()},
                                {"features"_L1, features}
                            }}};
    m_webSocket.sendTextMessage(QString::fromUtf8(QJsonDocument(json).toJson()));
}

void WebsocketClient::info(const QJsonObject &args)
{
    const auto email = args["email"_L1].toString();
    sendStatusUpdate(email); // web client expects to know that info before info-fetched
    const EwsId id(args["itemId"_L1].toString());
    if (m_cachedInfo.contains(id)) {
        m_webSocket.sendTextMessage(m_cachedInfo[id]);
        return;
    }

    EwsClient client(email, args["ewsAccessToken"_L1].toString().toUtf8());

    EwsItemShape itemShape(EwsShapeIdOnly);
    itemShape.setFlags(EwsItemShape::IncludeMimeContent);
    itemShape << EwsPropertyField(u"item:ParentFolderId"_s);

    auto request = new EwsGetItemRequest(client, this);
    request->setItemIds({id});
    request->setItemShape(itemShape);
    qCWarning(GPGOL_CLIENT_LOG) << "Info for" << id << "requested";
    connect(request, &EwsGetItemRequest::finished, this, [this, id, args, request]() {
        if (request->error() != KJob::NoError) {
            const QJsonObject json{{"command"_L1, Protocol::commandToString(Protocol::Error)},
                                   {"arguments"_L1,
                                    QJsonObject{
                                        {"error"_L1, request->errorString()},
                                        {"email"_L1, args["email"_L1]},
                                    }}};
            const auto result = QString::fromUtf8(QJsonDocument(json).toJson());
            m_webSocket.sendTextMessage(result);
            return;
        }

        qCWarning(GPGOL_CLIENT_LOG) << "Info for" << id << "fetched";
        const auto responses = request->responses();
        if (responses.isEmpty()) {
            return;
        }

        const auto item = responses.at(0).item();
        const auto mimeContent = item[EwsItemFieldMimeContent].toString();

        KMime::Message::Ptr message(new KMime::Message());
        message->setContent(KMime::CRLFtoLF(mimeContent.toUtf8()));
        message->parse();

        MimeTreeParser::ObjectTreeParser treeParser;
        treeParser.parseObjectTree(message.get());

        const QJsonObject json{{"command"_L1, Protocol::commandToString(Protocol::InfoFetched)},
                               {"arguments"_L1,
                                QJsonObject{
                                    {"itemId"_L1, args["itemId"_L1]},
                                    {"folderId"_L1, item[EwsItemFieldParentFolderId].value<EwsId>().id()},
                                    {"email"_L1, args["email"_L1]},
                                    {"encrypted"_L1, treeParser.hasEncryptedParts()},
                                    {"signed"_L1, treeParser.hasSignedParts()},
                                    {"version"_L1, QStringLiteral(GPGOLWEB_VERSION_STRING)},
                                }}};

        const auto result = QString::fromUtf8(QJsonDocument(json).toJson());
        m_cachedInfo[id] = result;
        m_cachedMime[id] = mimeContent;
        m_webSocket.sendTextMessage(result);
    });

    request->start();
}

QString WebsocketClient::getId() const
{
    auto config = KSharedConfig::openStateConfig();
    auto machineGroup = config->group(u"Machine"_s);
    if (machineGroup.exists() && machineGroup.hasKey(u"Id"_s)) {
        return machineGroup.readEntry(u"Id"_s);
    }

    const auto id = QUuid::createUuid().toString(QUuid::WithoutBraces);
    machineGroup.writeEntry("Id", id);
    config->sync();
    return id;
}
