// SPDX-FileCopyrightText: 2025 g10 code GmbH
// SPDX-Contributor: Carl Schwan <carl.schwan@gnupg.com>
// SPDX-License-Identifier: BSD-2-Clause

#include "truststore_linux_helper.h"

#include <QFileInfo>
#include <QProcess>
#include <QStandardPaths>

#include <KLocalizedString>

using namespace Qt::StringLiterals;

ActionReply Helper::install(const QVariantMap &args)
{
    ActionReply reply;

    if (!QStandardPaths::findExecutable(u"apt"_s).isEmpty()) {
        mCertutilInstallHelp = u"apt install libnss3-tools"_s;
    } else if (!QStandardPaths::findExecutable(u"yum"_s).isEmpty()) {
        mCertutilInstallHelp = u"yum install nss-tools"_s;
    } else if (!QStandardPaths::findExecutable(u"zypper"_s).isEmpty()) {
        mCertutilInstallHelp = u"zypper install mozilla-nss-tools"_s;
    }

    QString systemTrustFilenameTemplate;

    if (QFileInfo::exists(u"/etc/pki/ca-trust/source/anchors/"_s)) {
        systemTrustFilenameTemplate = u"/etc/pki/ca-trust/source/anchors/%1.pem"_s;
        mSystemTrustCommand = {u"update-ca-trust"_s, u"extract"_s};
    } else if (QFileInfo::exists(u"/usr/local/share/ca-certificates/"_s)) {
        systemTrustFilenameTemplate = u"/usr/local/share/ca-certificates/%1.crt"_s;
        mSystemTrustCommand = {u"update-ca-certificates"_s};
    } else if (QFileInfo::exists(u"/etc/ca-certificates/trust-source/anchors/"_s)) {
        systemTrustFilenameTemplate = u"/etc/ca-certificates/trust-source/anchors/%1.crt"_s;
        mSystemTrustCommand = {u"trust"_s, u"extract-compat"_s};
    } else if (QFileInfo::exists(u"/usr/share/pki/trust/anchors"_s)) {
        systemTrustFilenameTemplate = u"/usr/share/pki/trust/anchors/%1.pem"_s;
        mSystemTrustCommand = {u"update-ca-certificates"_s};
    }

    const auto systemTrustFilename = systemTrustFilenameTemplate.arg(args["caUniqueName"_L1].toString());

    QFile systemTrustFile(systemTrustFilename);
    if (!systemTrustFile.open(QIODeviceBase::WriteOnly)) {
        reply.setErrorCode(ActionReply::Error::InvalidActionError);
        reply.setErrorDescription(i18n("Unable to open %s", systemTrustFilename));
        return reply;
    }

    const auto caCert = args["caCert"_L1].toByteArray();

    systemTrustFile.write(caCert);
    systemTrustFile.close();

    QProcess proc;
    const QString command = mSystemTrustCommand.constFirst();
    mSystemTrustCommand.removeFirst();
    proc.start(command, mSystemTrustCommand);
    proc.waitForFinished();

    if (proc.exitCode() != 0) {
        reply.setErrorCode(ActionReply::Error::InvalidActionError);
        reply.setErrorDescription(u"Command failed wit exit code"_s + QString::fromUtf8(proc.readAll()));
        return false;
    }

    return reply;
}

KAUTH_HELPER_MAIN("com.gnupg.gpgolweb.truststore", Helper)
