// SPDX-FileCopyrightText: 2025 g10 code GmbH
// SPDX-FileContributor: Carl Schwan <carl.schwan@gnupg.com>
// SPDX-License-Identifier: GPL-2.0-or-later

#include "choosekeydialog.h"

#include "certificatelineedit.h"

#include <qdialogbuttonbox.h>
#include <ranges>

#include <QButtonGroup>
#include <QDialogButtonBox>
#include <QLabel>
#include <QPushButton>
#include <QRadioButton>
#include <QVBoxLayout>

#include <Libkleo/DefaultKeyFilter>
#include <Libkleo/KeyCache>
#include <Libkleo/KeyFilter>
#include <Libkleo/KeyGroup>
#include <Libkleo/KeyListModel>
#include <Libkleo/KeySelectionCombo>

#include "ui_choosekeydialog.h"
#include <KAdjustingScrollArea>
#include <KLocalizedString>
#include <KMessageWidget>
#include <KTitleWidget>

using namespace Qt::StringLiterals;

namespace
{

void forceSetTabOrder(QWidget *first, QWidget *second)
{
    if (!first || !second || first == second) {
        return;
    }
    // temporarily change the focus policy of the two widgets to something
    // other than Qt::NoFocus because QWidget::setTabOrder() does nothing
    // if either widget has focus policy Qt::NoFocus
    const auto firstFocusPolicy = first->focusPolicy();
    const auto secondFocusPolicy = second->focusPolicy();
    if (firstFocusPolicy == Qt::NoFocus) {
        first->setFocusPolicy(Qt::StrongFocus);
    }
    if (secondFocusPolicy == Qt::NoFocus) {
        second->setFocusPolicy(Qt::StrongFocus);
    }
    QWidget::setTabOrder(first, second);
    if (first->focusPolicy() != firstFocusPolicy) {
        first->setFocusPolicy(firstFocusPolicy);
    }
    if (second->focusPolicy() != secondFocusPolicy) {
        second->setFocusPolicy(secondFocusPolicy);
    }
}
}

ChooseKeyDialog::ChooseKeyDialog(const EwsFolder &ewsFolder, QWidget *parent)
    : QDialog(parent)
    , ui(std::make_unique<Ui::ChooseKeyDialog>())
    , m_ewsFolder(ewsFolder)
    , m_keyFilter(std::make_shared<Kleo::DefaultKeyFilter>())
    , m_model{Kleo::AbstractKeyListModel::createFlatKeyListModel(this)}
{
    ui->setupUi(this);

    ui->title->setText(i18nc("@title", "Selected folder: %1", m_ewsFolder[EwsItemFields::EwsFolderFieldDisplayName].toString()));

    m_model->useKeyCache(true, Kleo::KeyList::IncludeGroups);

    m_keyFilter->setIsOpenPGP(Kleo::DefaultKeyFilter::Set);
    m_keyFilter->setCanEncrypt(Kleo::DefaultKeyFilter::Set);
    m_keyFilter->setName(i18n("Usable for Encryption"));
    m_keyFilter->setDescription(i18n("Certificates that can be used for encryption"));

    m_recpLayout = new QVBoxLayout(ui->recpWidget);
    m_recpLayout->setContentsMargins({});

    addRecipientWidget();

    ui->folderLabel->setText(i18nc("@info",
                                   "Reencrypted emails will be saved to the new folder \"%1 - reencrypted\".",
                                   m_ewsFolder[EwsItemFields::EwsFolderFieldDisplayName].toString()));

    m_unencrypted = new QButtonGroup(this);
    m_unencrypted->addButton(ui->unencryptSkip, static_cast<int>(UnencryptedMode::Skip));
    ui->unencryptSkip->setChecked(true);
    m_unencrypted->addButton(ui->unencryptCopy, static_cast<int>(UnencryptedMode::Copy));

    resize(std::max(sizeHint().width(), 500) + 20, sizeHint().height() + 50);
    ui->buttonBox->button(QDialogButtonBox::Ok)->setEnabled(false);
}

ChooseKeyDialog::~ChooseKeyDialog() = default;

UnencryptedMode ChooseKeyDialog::unencryptedMode() const
{
    return static_cast<UnencryptedMode>(m_unencrypted->checkedId());
}

std::vector<GpgME::Key> ChooseKeyDialog::currentKeys() const
{
    std::vector<GpgME::Key> keys;

    for (const auto lineEdit : m_recpWidgets) {
        const GpgME::Key k = lineEdit->key();
        const Kleo::KeyGroup g = lineEdit->group();
        const GpgME::UserID u = lineEdit->userID();
        if (!k.isNull()) {
            keys.push_back(k);
        } else if (!g.isNull()) {
            const auto groupKeys = g.keys();
            std::copy(groupKeys.begin(), groupKeys.end(), std::back_inserter(keys));
        } else if (!u.isNull()) {
            keys.push_back(u.parent());
        }
    }

    auto last = std::unique(keys.begin(), keys.end(), [](const auto &key1, const auto &key2) {
        return strcmp(key1.primaryFingerprint(), key2.primaryFingerprint());
    });
    keys.erase(last, keys.end());

    return keys;
}

Kleo::CertificateLineEdit *ChooseKeyDialog::addRecipientWidget()
{
    return insertRecipientWidget(nullptr);
}

Kleo::CertificateLineEdit *ChooseKeyDialog::insertRecipientWidget(Kleo::CertificateLineEdit *after)
{
    Q_ASSERT(!after || m_recpLayout->indexOf(after) != -1);

    const auto index = after ? m_recpLayout->indexOf(after) + 1 : m_recpLayout->count();

    auto lineEdit = new Kleo::CertificateLineEdit(m_model, Kleo::KeyUsage::Encrypt, m_keyFilter, this);
    lineEdit->setKeyFilter(m_keyFilter);
    lineEdit->setEnabled(!Kleo::KeyCache::instance()->keys().empty());

    if (!after) {
        ui->recipientsLabel->setBuddy(lineEdit);
    }
    if (static_cast<unsigned>(index / 2) < m_recpWidgets.size()) {
        m_recpWidgets.insert(m_recpWidgets.begin() + index / 2, lineEdit);
    } else {
        m_recpWidgets.push_back(lineEdit);
    }

    if (m_recpLayout->count() > 0) {
        auto prevWidget = after ? after : m_recpLayout->itemAt(m_recpLayout->count() - 1)->widget();
        forceSetTabOrder(prevWidget, lineEdit);
    }
    m_recpLayout->insertWidget(index, lineEdit);

    connect(lineEdit, &Kleo::CertificateLineEdit::keyChanged, this, &ChooseKeyDialog::recipientsChanged);
    connect(lineEdit, &Kleo::CertificateLineEdit::editingStarted, this, &ChooseKeyDialog::recipientsChanged);
    connect(lineEdit, &Kleo::CertificateLineEdit::cleared, this, &ChooseKeyDialog::recipientsChanged);

    return lineEdit;
}

void ChooseKeyDialog::recipientsChanged()
{
    const bool hasEmptyRecpWidget = std::any_of(std::cbegin(m_recpWidgets), std::cend(m_recpWidgets), [](auto w) {
        return w->isEmpty();
    });

    if (hasEmptyRecpWidget) {
        bool foundEmpty = false;
        const auto lineEdits = m_recpWidgets;
        for (const auto lineEdit : lineEdits | std::views::reverse) {
            if (lineEdit->isEmpty() && !foundEmpty) {
                foundEmpty = true;
                continue;
            } else if (lineEdit->isEmpty()) {
                m_recpLayout->removeWidget(lineEdit);
                const auto it = std::find_if(std::begin(m_recpWidgets), std::end(m_recpWidgets), [lineEdit](const auto &r) {
                    return lineEdit == r;
                });
                m_recpWidgets.erase(it);
                lineEdit->deleteLater();
            }
        }
    } else {
        addRecipientWidget();
    }
    const bool hasAtLeastSomeContent = std::any_of(std::cbegin(m_recpWidgets), std::cend(m_recpWidgets), [](auto w) {
        return !w->isEmpty();
    });
    ui->buttonBox->button(QDialogButtonBox::Ok)->setEnabled(hasAtLeastSomeContent);
}
